import math

# -------------------------
# Configuration Parameters
# -------------------------
SCALE = 10**10
MAX_RDIM = 255
SEED_VECTORS = 4            # minimal on-chain seed vectors
FOLD_DOUBLINGS = [2]*24     # doubling to reach 16,777,216 vectors
CTv_PUBKEY = "<YOUR_PUBLIC_KEY>"

FIBS = [1,1,2,3,5,8,13,21,34,55,89,144,233,377,610]

# Seed params for integrity (hex string)
SEED_PARAMS = [
    ("0", "0x812"), 
    ("1", "0x502"),
    ("2", "0x310"),
    ("3", "0x192")
]

# -------------------------
# Covenant Generator
# -------------------------
def generate_shared_covenant():
    script = []
    script.append(f"{CTv_PUBKEY} OP_CHECKSIGVERIFY")

    vector_count = SEED_VECTORS
    fib_index = 0

    # Loop over folding doublings
    for fold in FOLD_DOUBLINGS:
        prev_count = vector_count
        vector_count *= fold
        script.append(f"OP_VEC_DUP {prev_count} OP_HDGL_VEC_ADD OP_EQUALVERIFY")
        script.append(f"{prev_count} {prev_count} OP_HDGL_PARAM_ADD OP_EQUALVERIFY")

        # Fibonacci delta
        fib_delta = FIBS[fib_index % len(FIBS)]
        fib_index += 1

        # Select seed param
        param = SEED_PARAMS[(vector_count.bit_length()-1) % len(SEED_PARAMS)]
        omega_int = int(param[1], 16)
        omega_min = omega_int + fib_delta
        omega_max = omega_int + fib_delta
        script.append(f"{omega_int} {omega_min} {omega_max} OP_WITHINVERIFY")
        script.append("")

    script.append("OP_CHECKTEMPLATEVERIFY")
    return '\n'.join(script)

# -------------------------
# Off-Chain Workspace Generator
# -------------------------
def generate_user_workspace(user_id, total_vectors=16_777_216):
    """
    Generate private mapping for a user.
    Each vector index can have user-specific parameters.
    """
    workspace = {}
    for idx in range(total_vectors):
        # Example: assign r_dim and omega differently per user
        r_dim = ((idx + user_id) % 256) / 255  # scaled to [0,1]
        omega = 1e-8 + ((idx * 7 + user_id) % 1000) * 1e-12
        workspace[idx] = {
            "r_dim": r_dim,
            "omega": omega,
            "private_key": f"user_{user_id}_vec_{idx}"  # optional per-vector private seed
        }
    return workspace

# -------------------------
# Example Usage
# -------------------------
if __name__ == "__main__":
    # 1. Generate on-chain covenant
    covenant_script = generate_shared_covenant()
    print("=== Generated Shared HDGL Covenant ===")
    print(covenant_script)
    print("\n\n")

    # 2. Generate off-chain workspace for user 1
    user1_workspace = generate_user_workspace(user_id=1)
    print("=== Sample User 1 Workspace Vectors ===")
    # Show first 10 vectors
    for idx in range(10):
        print(f"Vector {idx}: {user1_workspace[idx]}")
